<?php defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Language Class
 *  
 */
class MY_Lang extends CI_Lang {

	/**
	 * List of translations
	 *
	 * @var	array
	 */
	public $language =	array();

	/**
	 * List of loaded language files
	 *
	 * @var	array
	 */
	public $is_loaded =	array();

	/**
	 * Class constructor
	 *
	 * @return	void
	 */
	public function __construct()
	{
		log_message('info', 'Language Class Initialized');
	}

	// --------------------------------------------------------------------

	/**
	 * Load a language file
	 *
	 * @param	mixed	$langfile	Language file name
	 * @param	string	$idiom		Language name (english, etc.)
	 * @param	bool	$return		Whether to return the loaded array of translations
	 * @param 	bool	$add_suffix	Whether to add suffix to $langfile
	 * @param 	string	$alt_path	Alternative path to look for the language file
	 *
	 * @return	void|string[]	Array containing translations, if $return is set to TRUE
	 */
	public function load($langfile, $idiom = '', $return = FALSE, $add_suffix = TRUE, $alt_path = '')
	{
		if (is_array($langfile))
		{
			foreach ($langfile as $value)
			{
				$this->load($value, $idiom, $return, $add_suffix, $alt_path);
			}

			return;
		}

		$config =& get_config();
		$langfile = str_replace('.php', '', $langfile);

		if ($add_suffix === TRUE)
		{
			$langfile = preg_replace('/_lang$/', '', $langfile).'_lang';
		}

		$langfile .= '.php';

		if (empty($idiom) OR ! preg_match('/^[a-z_-]+$/i', $idiom))
		{			
			$idiom = empty($config['language']) ? 'english' : $config['language'];
		}

		if ($return === FALSE && isset($this->is_loaded[$langfile]) && $this->is_loaded[$langfile] === $idiom)
		{
			return;
		}

		$found = FALSE;
		$lang = array();

		// 1. Load system language file first (BASEPATH)
		$basepath = BASEPATH.'language/'.$idiom.'/'.$langfile;
		if (file_exists($basepath))
		{
			include($basepath);
			$found = TRUE;
		}

		// 2. Load application language file (APPPATH) - this will override system
		$apppath = APPPATH.'language/'.$idiom.'/'.$langfile;
		if (file_exists($apppath))
		{
			include($apppath);
			$found = TRUE;
		}

		// 3. Load user custom language file (userdata_path) - this will override application
		if(empty($alt_path))
		{
			$alt_path = empty($config['userdata_path']) ? '' : $config['userdata_path'];
		}

		if ($alt_path !== '')
		{
			$userpath = $alt_path.'/language/'.$idiom.'/'.$langfile;
			if (file_exists($userpath))
			{
				include($userpath);
				$found = TRUE;
			}
		}
				
		// 4. If still not found, check package paths
		if($found !== TRUE)
		{
			foreach (get_instance()->load->get_package_paths(TRUE) as $package_path)
			{
				$package_path .= 'language/'.$idiom.'/'.$langfile;
				if ($basepath !== $package_path && file_exists($package_path))
				{					
					include($package_path);
					$found = TRUE;
					break;
				}
			}
		}

		if ($found !== TRUE)
		{
			//show_error('Unable to load the requested language file: language/'.$idiom.'/'.$langfile);
			log_message('info', 'Unable to load the requested language file: language/'.$idiom.'/'.$langfile);
			return TRUE;
		}

		if ( ! isset($lang) OR ! is_array($lang))
		{
			log_message('info', 'Language file contains no data: language/'.$idiom.'/'.$langfile);

			if ($return === TRUE)
			{
				return array();
			}
			return;
		}

		if ($return === TRUE)
		{
			return $lang;
		}

		$this->is_loaded[$langfile] = $idiom;
		$this->language = array_merge($this->language, $lang);

		log_message('info', 'Language file loaded: language/'.$idiom.'/'.$langfile);
		return TRUE;
	}


	/**
	 * Language line
	 *
	 * Fetches a single line of text from the language array
	 *
	 * @param	string	$line		Language line key
	 * @param	bool	$log_errors	Whether to log an error message if the line is not found
	 * @return	string	Translation
	 */
	public function line($line, $log_errors = TRUE)
	{
		$value = isset($this->language[$line]) ? $this->language[$line] : FALSE;

		// Because killer robots like unicorns!
		if ($value === FALSE && $log_errors === TRUE)
		{
			log_message('info', 'Could not find the language line "'.$line.'"');
		}

		return $value;
	}

}
